//	TorusGamesLanguageChoiceController.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesLanguageChoiceController.h"
#import "TorusGames-Common.h"	//	for gLanguages
#import "GeometryGamesUtilities-Common.h"
#import "GeometryGamesUtilities-iOS.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesLocalization.h"


@implementation TorusGamesLanguageChoiceController
{
	id<TorusGamesLanguageChoiceDelegate> __weak	itsDelegate;
	Char16										itsLanguageChoice[3];
	UIBarButtonItem								*itsCancelButton;
}

- (id)initWithDelegate:(id<TorusGamesLanguageChoiceDelegate>)aDelegate languageChoice:(const Char16 [3])aTwoLetterLanguageCode
{
	self = [super initWithStyle:UITableViewStyleGrouped];
	if (self != nil)
	{
		itsDelegate = aDelegate;

		itsLanguageChoice[0] = aTwoLetterLanguageCode[0];
		itsLanguageChoice[1] = aTwoLetterLanguageCode[1];
		itsLanguageChoice[2] = 0;

		[[self navigationItem] setTitle:GetLocalizedTextAsNSString(u"Language")];
		
		//	To make it easy to try Crossword and Word Search puzzles
		//	in different languages, Torus Games lets the player change
		//	languages on the fly.  So let's provide an explicit localization
		//	for the "Cancel" button, in lieu of the built-in
		//
		//		initWithBarButtonSystemItem:UIBarButtonSystemItemCancel
		//
		itsCancelButton = [[UIBarButtonItem alloc]
			initWithTitle:	GetLocalizedTextAsNSString(u"Cancel")
			style:			UIBarButtonItemStylePlain
			target:			aDelegate
			action:			@selector(userDidCancelLanguageChoice)];
	}
	return self;
}

- (BOOL)prefersStatusBarHidden
{
	return [itsDelegate prefersStatusBarHidden];
}

- (void)viewWillAppear:(BOOL)animated
{
	[super viewWillAppear:animated];

	//	We must call -layoutIfNeeded explicitly, otherwise -contentSize returns CGSizeZero.
	[[self tableView] layoutIfNeeded];
	[self setPreferredContentSize:(CGSize){	PREFERRED_POPOVER_WIDTH,
											[[self tableView] contentSize].height}];

	[self adaptNavBarForHorizontalSize:
		[[RootViewController(self) traitCollection] horizontalSizeClass]];
}


#pragma mark -
#pragma mark GeometryGamesPopover

- (void)adaptNavBarForHorizontalSize:(UIUserInterfaceSizeClass)aHorizontalSizeClass
{
	[[self navigationItem]
		setRightBarButtonItem:	(aHorizontalSizeClass == UIUserInterfaceSizeClassCompact ?
									itsCancelButton : nil)
		animated:				NO];
}


#pragma mark -
#pragma mark UITableViewDataSource

- (NSInteger)numberOfSectionsInTableView:(UITableView *)aTableView
{
	UNUSED_PARAMETER(aTableView);

	return 1;
}

- (NSString *)tableView:(UITableView *)aTableView titleForHeaderInSection:(NSInteger)aSection
{
	UNUSED_PARAMETER(aTableView);
	UNUSED_PARAMETER(aSection);

	//	The section needs no title.
	return nil;
}

- (NSInteger)tableView:(UITableView *)aTableView numberOfRowsInSection:(NSInteger)aSection
{
	UNUSED_PARAMETER(aTableView);
	UNUSED_PARAMETER(aSection);

	return gNumLanguages;
}

- (UITableViewCell *)tableView:(UITableView *)aTableView cellForRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	UITableViewCell	*theCell;
	UILabel			*theLabel;
	NSUInteger		theRow;
	const Char16	*theEndonym;
	NSString		*theTitle;
	bool			theCheckmark;

	UNUSED_PARAMETER(aTableView);

	theCell			= [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:nil];
	theLabel		= [theCell textLabel];
	theRow			= [anIndexPath row];
	theEndonym		= GetEndonym(gLanguages[theRow]);
	theTitle		= GetNSStringFromZeroTerminatedString(theEndonym);
	theCheckmark	= (gLanguages[theRow][0] == itsLanguageChoice[0]
					&& gLanguages[theRow][1] == itsLanguageChoice[1]);

	[theLabel setText:theTitle];
	[theCell setAccessoryType:(theCheckmark ? UITableViewCellAccessoryCheckmark : UITableViewCellAccessoryNone)];
	[[theCell imageView] setImage:[UIImage imageNamed:
		[NSString stringWithFormat:@"Table Images/Language/%c%c",
			gLanguages[theRow][0],
			gLanguages[theRow][1]]]];

	return theCell;
}


#pragma mark -
#pragma mark UITableViewDelegate

- (NSIndexPath *)tableView:(UITableView *)aTableView willSelectRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	NSUInteger	theRow;

	UNUSED_PARAMETER(aTableView);

	theRow = [anIndexPath row];

	itsLanguageChoice[0] = gLanguages[theRow][0];
	itsLanguageChoice[1] = gLanguages[theRow][1];
	itsLanguageChoice[2] = 0;

	//	Reload the table so the checkmark will appear
	//	by the newly selected language.  The user will
	//	see it briefly as the table animates away.
	//	The effect is more noticeable on an iPad,
	//	where the popover fades away in place, than
	//	on an iPod touch or iPhone, where the presented view
	//	slides off the bottom of the screen.
	[[self tableView] reloadData];

	//	Report the new language to the caller,
	//	who will dismiss this TorusGamesLanguageChoiceController.
	[itsDelegate userDidChooseLanguage:itsLanguageChoice];

	return nil;
}

@end
